//
//  DataCenter.swift
//  Do It
//
//  Created by Jim Dovey on 10/13/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import Foundation
import Combine

fileprivate let jsonDataName = "todoData.json"

class DataCenter: ObservableObject {
    @Published var todoItems: [TodoItem] = []
    @Published var todoLists: [TodoItemList] = []
    private var saveCancellable: AnyCancellable?
    
    init() {
        let todoData: TodoData
        do {
            todoData = try loadJSON(from: jsonDataURL)
        } catch {
            todoData = createDefaultItems()
        }
        self.todoItems = todoData.items
        self.todoLists = todoData.lists
        saveWhenChanged()
    }

    private func saveWhenChanged() {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601

        let scheduler = UIBackgroundTaskScheduler(
            "Saving To-Do Items", target: DispatchQueue.global())

        self.saveCancellable = $todoItems
            .combineLatest($todoLists) { TodoData(lists: $1, items: $0) }
            .dropFirst()
            .debounce(for: .milliseconds(100), scheduler: scheduler)
            .encode(encoder: encoder)
            .sink(receiveCompletion: { _ in }, receiveValue: self.writeData)
    }
    
    func list(for item: TodoItem) -> TodoItemList {
        guard let list = todoLists.first(where: { $0.id == item.listID }) else {
            fatalError("No matching list for ID: \(item.listID)")
        }
        return list
    }
    
    func items(in list: TodoItemList) -> [TodoItem] {
        todoItems.filter { $0.listID == list.id }
    }

    private func createDefaultItems() -> TodoData {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601
        do {
            let data = try encoder.encode(defaultTodoData)
            writeData(data)
        } catch {
            fatalError("Failed to create default todo items! \(error)")
        }
        return defaultTodoData
    }

    private func writeData(_ data: Data) {
        do {
            try data.write(to: self.jsonDataURL,
                           options: [.completeFileProtection, .atomic])
        } catch {
            NSLog("Error writing JSON data: \(error)")
        }
    }

    private func loadJSON<T: Decodable>(
        from url: URL,
        as type: T.Type = T.self
    ) throws -> T {
        let data = try Data(contentsOf: url,
                            options: [.mappedIfSafe])
        
        let decoder = JSONDecoder()
        decoder.dateDecodingStrategy = .iso8601
        return try decoder.decode(type, from: data)
    }

    private lazy var jsonDataURL: URL = {
        let baseURL: URL
        do {
            baseURL = try FileManager.default.url(
                for: .documentDirectory, in: .userDomainMask,
                appropriateFor: nil, create: true)
        } catch {
            let homeURL = URL(fileURLWithPath: NSHomeDirectory(),
                              isDirectory: true)
            baseURL = homeURL.appendingPathComponent("Documents")
        }
        return baseURL.appendingPathComponent(jsonDataName)
    }()
}
